
console.log('%c🍓 Berry Ave Whitelist Tool: Script injected!', 'color: #667eea; font-weight: bold; font-size: 14px');

const WORKER_URL = 'https://berryitems.com';
const RATE_LIMIT_DELAY = 1000; 
const MAX_RETRY_ATTEMPTS = 3;
const RETRY_DELAY = 3000; 

let whitelistedItems = new Set(); 
let apiKey = null;
let processingQueue = [];
let isProcessingQueue = false;
let failedItems = new Map(); 
let removingItems = new Set(); 

(async function init() {
  console.log('%c🍓 Berry Ave: Initializing...', 'color: #667eea; font-weight: bold');
  
  try {
    const result = await chrome.storage.local.get(['apiKey']);
    apiKey = result.apiKey;
    console.log('🍓 Berry Ave: API key loaded:', apiKey ? 'Yes' : 'No');
  } catch (error) {
    console.error('🍓 Berry Ave: Error loading API key:', error);
  }
  
  setTimeout(waitForItems, 1000);
})();

function waitForItems() {
  console.log('🍓 Berry Ave: Looking for catalog items...');
  
  const catalogContainers = document.querySelectorAll('.catalog-item-container');
  const itemCards = document.querySelectorAll('.item-card-container');
  
  console.log(`🍓 Berry Ave: Found ${catalogContainers.length} catalog containers, ${itemCards.length} item cards`);
  
  if (catalogContainers.length > 0 || itemCards.length > 0) {
    console.log('🍓 Berry Ave: Items found! Injecting UI...');
    injectUI();
    observePageChanges();
    checkWhitelistedItems();
  } else {
    console.log('🍓 Berry Ave: No items yet, retrying in 500ms...');
    setTimeout(waitForItems, 500);
  }
}

async function checkWhitelistedItems() {
  if (!apiKey) {
    console.log('🍓 Berry Ave: No API key, skipping whitelist check');
    return;
  }
  
  const itemIds = [];
  document.querySelectorAll('.catalog-item-container').forEach(container => {
    const itemData = extractItemData(container);
    if (itemData && itemData.id) {
      itemIds.push(itemData.id);
    }
  });
  
  if (itemIds.length === 0) return;
  
  console.log(`🍓 Berry Ave: Checking whitelist status for ${itemIds.length} items...`);
  
  try {
    const response = await fetch(`${WORKER_URL}/api/items/check`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-Key': apiKey
      },
      body: JSON.stringify({ itemIds })
    });
    
    if (response.ok) {
      const data = await response.json();
      if (data.whitelisted && Array.isArray(data.whitelisted)) {
        whitelistedItems = new Set(data.whitelisted);
        console.log(`🍓 Berry Ave: Found ${whitelistedItems.size} whitelisted items`);
        
        updateWhitelistedIndicators();
      }
    }
  } catch (error) {
    console.error('🍓 Berry Ave: Error checking whitelisted items:', error);
  }
}

function updateWhitelistedIndicators() {
  document.querySelectorAll('.catalog-item-container').forEach(container => {
    const itemData = extractItemData(container);
    if (!itemData) return;
    
    const isWhitelisted = whitelistedItems.has(itemData.id);
    
    if (isWhitelisted) {
      applyWhitelistedOverlay(itemData.id);
    }
  });
}

function applyWhitelistedOverlay(itemId) {
  const containers = document.querySelectorAll('.catalog-item-container');
  
  containers.forEach(container => {
    const itemData = extractItemData(container);
    if (!itemData || itemData.id !== itemId) return;
    
    container.style.position = 'relative';
    
    const itemCard = container.querySelector('.item-card-container');
    if (itemCard) {
      itemCard.style.border = '3px solid #10b981';
      itemCard.style.borderRadius = '8px';
      itemCard.style.transition = 'all 0.3s';
    }
    
    const existingBadge = container.querySelector('.berry-whitelisted-badge');
    if (existingBadge) {
      existingBadge.remove();
    }
    
    const badge = document.createElement('div');
    badge.className = 'berry-whitelisted-badge';
    badge.innerHTML = `
      <span class="berry-badge-text">✓ Whitelisted</span>
      <button class="berry-remove-btn" data-item-id="${itemId}" title="Remove from whitelist">×</button>
    `;
    badge.style.cssText = `
      position: absolute;
      top: 10px;
      right: 10px;
      background: linear-gradient(135deg, #10b981, #059669);
      color: white;
      padding: 6px 12px;
      border-radius: 20px;
      font-size: 11px;
      font-weight: 600;
      z-index: 100;
      box-shadow: 0 2px 8px rgba(16, 185, 129, 0.3);
      display: flex;
      align-items: center;
      gap: 8px;
      opacity: 0;
      transform: scale(0.8);
      transition: all 0.3s;
    `;
    
    container.appendChild(badge);
    
    const removeBtn = badge.querySelector('.berry-remove-btn');
    if (removeBtn) {
      removeBtn.style.cssText = `
        background: rgba(255, 255, 255, 0.2);
        border: 1px solid rgba(255, 255, 255, 0.3);
        color: white;
        width: 20px;
        height: 20px;
        border-radius: 50%;
        font-size: 16px;
        line-height: 1;
        cursor: pointer;
        display: none;
        transition: all 0.2s;
        padding: 0;
        margin: 0;
      `;
      
      badge.addEventListener('mouseenter', () => {
        if (!removingItems.has(itemId)) {
          removeBtn.style.display = 'flex';
          removeBtn.style.alignItems = 'center';
          removeBtn.style.justifyContent = 'center';
        }
      });
      
      badge.addEventListener('mouseleave', () => {
        if (!removingItems.has(itemId)) {
          removeBtn.style.display = 'none';
        }
      });
      
      removeBtn.addEventListener('click', async (e) => {
        e.stopPropagation();
        e.preventDefault();
        
        if (removingItems.has(itemId)) return;
        
        if (!apiKey) {
          showNotification('Please set your API key in the extension popup first!', 'error');
          return;
        }
        
        if (!confirm(`Remove "${itemData.itemName}" from whitelist?`)) {
          return;
        }
        
        removingItems.add(itemId);
        
        badge.style.background = 'linear-gradient(135deg, #fbbf24, #f59e0b)';
        const badgeText = badge.querySelector('.berry-badge-text');
        if (badgeText) {
          badgeText.innerHTML = '<span class="berry-loading-spinner"></span>Removing...';
        }
        removeBtn.style.display = 'none';
        
        try {
          const response = await fetch(`${WORKER_URL}/api/items/remove`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'X-API-Key': apiKey
            },
            body: JSON.stringify({ 
              itemIds: [itemId],
              reason: 'Removed via extension'
            })
          });
          
          if (!response.ok) {
            const error = await response.json();
            throw new Error(error.error || 'Failed to remove item');
          }
          
          const result = await response.json();
          
          whitelistedItems.delete(itemId);
          removingItems.delete(itemId);
          
          badge.remove();
          if (itemCard) {
            itemCard.style.border = '';
          }
          
          // Show the whitelist checkbox again
          const wrapper = container.querySelector('.berry-checkbox-wrapper');
          if (wrapper) {
            wrapper.style.display = 'flex';
            wrapper.classList.remove('success', 'processing', 'error');
            const checkbox = wrapper.querySelector('.berry-checkbox');
            const label = wrapper.querySelector('label');
            if (checkbox) {
              checkbox.checked = false;
              checkbox.disabled = false;
            }
            if (label) {
              label.textContent = 'Whitelist';
            }
          } else {
            addCheckboxToContainer(container);
          }
          
          updateStatsDisplay();
          
          showNotification(`Removed "${itemData.itemName}" from whitelist`, 'success');
          console.log(`🍓 Berry Ave: Successfully removed ${itemData.itemName} from whitelist`);
          
        } catch (error) {
          console.error(`🍓 Berry Ave: Error removing ${itemData.itemName}:`, error);
          removingItems.delete(itemId);
          
          // Revert UI state
          badge.style.background = 'linear-gradient(135deg, #10b981, #059669)';
          if (badgeText) {
            badgeText.textContent = '✓ Whitelisted';
          }
          
          showNotification(`Failed to remove "${itemData.itemName}": ${error.message}`, 'error');
        }
      });
      
      removeBtn.addEventListener('mouseenter', () => {
        removeBtn.style.background = 'rgba(239, 68, 68, 0.8)';
        removeBtn.style.transform = 'scale(1.1)';
      });
      
      removeBtn.addEventListener('mouseleave', () => {
        removeBtn.style.background = 'rgba(255, 255, 255, 0.2)';
        removeBtn.style.transform = 'scale(1)';
      });
    }
    
    setTimeout(() => {
      badge.style.opacity = '1';
      badge.style.transform = 'scale(1)';
    }, 50);
    
    const wrapper = container.querySelector('.berry-checkbox-wrapper');
    if (wrapper && !wrapper.classList.contains('success')) {
      wrapper.style.display = 'none';
    }
  });
}

function addCheckboxToContainer(container) {
  if (container.querySelector('.berry-checkbox-wrapper')) return;
  
  const itemData = extractItemData(container);
  if (!itemData) return;
  
  if (whitelistedItems.has(itemData.id)) return;
  
  let targetContainer = container;
  const innerContainer = container.querySelector('.item-card-container');
  if (innerContainer) {
    targetContainer = innerContainer;
  }
  
  const wrapper = document.createElement('div');
  wrapper.className = 'berry-checkbox-wrapper';
  wrapper.style.cssText = `
    position: absolute;
    top: 10px;
    left: 10px;
    z-index: 100;
    background: white;
    border-radius: 8px;
    padding: 8px 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.15);
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
    transition: all 0.2s;
    min-width: 100px;
  `;
  
  const checkbox = document.createElement('input');
  checkbox.type = 'checkbox';
  checkbox.className = 'berry-checkbox';
  checkbox.dataset.itemId = itemData.id;
  checkbox.style.cssText = `
    width: 18px;
    height: 18px;
    cursor: pointer;
    margin: 0;
  `;
  
  const label = document.createElement('label');
  label.textContent = 'Whitelist';
  label.style.cssText = `
    font-size: 12px;
    font-weight: 600;
    color: #333;
    cursor: pointer;
    user-select: none;
    margin: 0;
  `;
  
  // Build the wrapper
  wrapper.appendChild(checkbox);
  wrapper.appendChild(label);
  
  if (getComputedStyle(targetContainer).position === 'static') {
    targetContainer.style.position = 'relative';
  }
  
  targetContainer.appendChild(wrapper);
  
  checkbox.berryItemData = itemData;
  
  wrapper.addEventListener('click', async function(e) {
    e.preventDefault();
    e.stopPropagation();
    
    if (wrapper.classList.contains('processing') || wrapper.classList.contains('success')) {
      return;
    }
    
    if (!apiKey) {
      showNotification('Please set your API key in the extension popup first!', 'error');
      return;
    }
    
    if (e.target !== checkbox) {
      checkbox.checked = !checkbox.checked;
    }
    
    if (checkbox.checked) {
      addToProcessingQueue(itemData, wrapper, checkbox, label);
    } else {
      const queueIndex = processingQueue.findIndex(item => item.data.id === itemData.id && !item.processed);
      if (queueIndex !== -1) {
        processingQueue.splice(queueIndex, 1);
        wrapper.classList.remove('processing');
        label.textContent = 'Whitelist';
      }
    }
  }, true);
  
  wrapper.addEventListener('mousedown', function(e) {
    e.stopPropagation();
  }, true);
}

function injectUI() {
  console.log('🍓 Berry Ave: Starting UI injection...');
  
  if (!document.getElementById('berry-ave-styles')) {
    const styleSheet = document.createElement('style');
    styleSheet.id = 'berry-ave-styles';
    styleSheet.textContent = `
      @keyframes berry-pulse {
        0% { transform: scale(1); }
        50% { transform: scale(1.05); }
        100% { transform: scale(1); }
      }
      
      @keyframes berry-spin {
        to { transform: rotate(360deg); }
      }
      
      .berry-whitelisted-badge {
        animation: berry-pulse 2s infinite;
      }
      
      .berry-checkbox-wrapper {
        transition: all 0.2s;
      }
      
      .berry-checkbox-wrapper.processing {
        background: linear-gradient(135deg, #fbbf24 0%, #f59e0b 100%) !important;
      }
      
      .berry-checkbox-wrapper.processing label {
        color: white !important;
      }
      
      .berry-checkbox-wrapper.success {
        background: linear-gradient(135deg, #10b981 0%, #059669 100%) !important;
        pointer-events: none;
      }
      
      .berry-checkbox-wrapper.success label {
        color: white !important;
      }
      
      .berry-checkbox-wrapper.error {
        background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%) !important;
      }
      
      .berry-checkbox-wrapper.error label {
        color: white !important;
      }
      
      .berry-checkbox-wrapper:hover:not(.processing):not(.success) {
        transform: scale(1.05);
        box-shadow: 0 4px 12px rgba(0,0,0,0.2);
      }
      
      .berry-loading-spinner {
        display: inline-block;
        width: 14px;
        height: 14px;
        border: 2px solid #ffffff;
        border-top-color: transparent;
        border-radius: 50%;
        animation: berry-spin 0.8s linear infinite;
        margin-right: 6px;
      }
      
      .berry-queue-indicator {
        position: fixed;
        bottom: 30px;
        right: 30px;
        background: white;
        padding: 12px 20px;
        border-radius: 25px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        font-weight: 600;
        color: #333;
        display: none;
        z-index: 9999;
      }
      
      .berry-queue-indicator.show {
        display: flex;
        align-items: center;
        gap: 10px;
      }
      
      .berry-retry-button {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        border: none;
        padding: 8px 16px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
        cursor: pointer;
        margin-left: 10px;
      }
      
      .berry-retry-button:hover {
        transform: scale(1.05);
      }
      
      .berry-remove-btn {
        display: flex;
        align-items: center;
        justify-content: center;
      }
      
      .berry-badge-text {
        display: flex;
        align-items: center;
        gap: 4px;
      }
    `;
    document.head.appendChild(styleSheet);
  }
  
  // Add queue indicator
  if (!document.getElementById('berry-queue-indicator')) {
    const queueIndicator = document.createElement('div');
    queueIndicator.id = 'berry-queue-indicator';
    queueIndicator.className = 'berry-queue-indicator';
    queueIndicator.innerHTML = `
      <span class="berry-loading-spinner"></span>
      <span id="berry-queue-text">Processing queue...</span>
    `;
    document.body.appendChild(queueIndicator);
  }
  
  // Add stats display (simplified)
  if (!document.getElementById('berry-ave-stats')) {
    const statsDisplay = document.createElement('div');
    statsDisplay.id = 'berry-ave-stats';
    statsDisplay.style.cssText = `
      position: fixed;
      bottom: 30px;
      left: 30px;
      background: white;
      padding: 12px 20px;
      border-radius: 25px;
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
      font-size: 12px;
      color: #666;
      z-index: 9999;
      display: none;
    `;
    statsDisplay.innerHTML = `<span id="berry-whitelisted-count">0</span> whitelisted on this page`;
    document.body.appendChild(statsDisplay);
  }
  
  console.log('🍓 Berry Ave: Adding checkboxes to items...');
  addItemCheckboxes();
  
  updateWhitelistedIndicators();
}

// Add checkboxes to catalog items
function addItemCheckboxes() {
  const itemContainers = document.querySelectorAll('.catalog-item-container');
  
  console.log(`🍓 Berry Ave: Processing ${itemContainers.length} items`);
  
  let addedCount = 0;
  let whitelistedCount = 0;
  
  itemContainers.forEach((container, index) => {
    // Skip if already has checkbox
    if (container.querySelector('.berry-checkbox-wrapper')) return;
    
    // Extract item data
    const itemData = extractItemData(container);
    if (!itemData) {
      console.log(`🍓 Berry Ave: Could not extract data for item ${index}`);
      return;
    }
    
    console.log(`🍓 Berry Ave: Processing item: ${itemData.itemName} (${itemData.id})`);
    
    const isWhitelisted = whitelistedItems.has(itemData.id);
    if (isWhitelisted) {
      whitelistedCount++;
      console.log(`🍓 Berry Ave: Skipping checkbox for whitelisted item: ${itemData.itemName}`);
      return;
    }
    
    addCheckboxToContainer(container);
    addedCount++;
  });
  
  console.log(`🍓 Berry Ave: Added ${addedCount} checkboxes, ${whitelistedCount} already whitelisted`);
  
  updateStatsDisplay();
}

function updateStatsDisplay() {
  const whitelistedCount = whitelistedItems.size;
  const statsDisplay = document.getElementById('berry-ave-stats');
  const countSpan = document.getElementById('berry-whitelisted-count');
  if (statsDisplay && countSpan) {
    countSpan.textContent = whitelistedCount;
    if (whitelistedCount > 0) {
      statsDisplay.style.display = 'block';
    } else {
      statsDisplay.style.display = 'none';
    }
  }
}

function addToProcessingQueue(itemData, wrapper, checkbox, label) {
  const queueItem = {
    data: itemData,
    wrapper: wrapper,
    checkbox: checkbox,
    label: label,
    processed: false,
    attempts: 0
  };
  
  processingQueue.push(queueItem);
  
  wrapper.classList.add('processing');
  label.innerHTML = '<span class="berry-loading-spinner"></span>Queued...';
  
  if (!isProcessingQueue) {
    processQueue();
  }
  
  updateQueueIndicator();
}

async function processQueue() {
  if (isProcessingQueue) return;
  
  isProcessingQueue = true;
  
  while (processingQueue.length > 0) {
    const item = processingQueue.find(i => !i.processed);
    if (!item) break;
    
    item.processed = true;
    item.attempts++;
    
    item.label.innerHTML = '<span class="berry-loading-spinner"></span>Processing...';
    updateQueueIndicator();
    
    try {
      console.log(`🍓 Berry Ave: Sending item ${item.data.id} to backend...`);

      // Send item data to berryitems.com backend
      const response = await fetch(`${WORKER_URL}/api/items/approve`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-API-Key': apiKey
        },
        body: JSON.stringify({
          items: [{
            id: item.data.id,
            robloxItemId: item.data.robloxItemId,
            itemName: item.data.itemName,
            itemType: item.data.itemType,
            price: item.data.price,
            thumbnailUrl: item.data.thumbnailUrl,
            itemUrl: item.data.itemUrl,
            creatorName: item.data.creatorName
          }]
        })
      });

      if (!response.ok) {
        const error = await response.json();
        console.error('🍓 Berry Ave: Backend error:', error);
        throw new Error(error.error || 'Failed to approve item');
      }
      
      const result = await response.json();
      console.log(`🍓 Berry Ave: Item ${item.data.id} approved successfully`);

      item.wrapper.style.display = 'none';
      
      whitelistedItems.add(item.data.id);
      
      applyWhitelistedOverlay(item.data.id);
      
      updateStorageStats(1);
      updateStatsDisplay();
      
      const index = processingQueue.indexOf(item);
      if (index > -1) {
        processingQueue.splice(index, 1);
      }
      
      failedItems.delete(item.data.id);
      
      console.log(`🍓 Berry Ave: Successfully whitelisted ${item.data.itemName}`);
      
    } catch (error) {
      console.error(`🍓 Berry Ave: Error whitelisting ${item.data.itemName}:`, error);

      item.wrapper.classList.remove('processing');
      item.wrapper.classList.add('error');

      if (item.attempts < MAX_RETRY_ATTEMPTS) {
        failedItems.set(item.data.id, item);
        item.label.innerHTML = `❌ Failed (Retry ${item.attempts}/${MAX_RETRY_ATTEMPTS})`;
        item.processed = false;

        setTimeout(() => {
          if (failedItems.has(item.data.id)) {
            item.wrapper.classList.remove('error');
            item.wrapper.classList.add('processing');
            item.label.innerHTML = '<span class="berry-loading-spinner"></span>Retrying...';
            processQueue();
          }
        }, RETRY_DELAY);
      } else {
        item.label.innerHTML = '❌ Failed';
        showNotification(`Failed to whitelist ${item.data.itemName} after ${MAX_RETRY_ATTEMPTS} attempts: ${error.message}`, 'error');

        item.checkbox.checked = false;

        const index = processingQueue.indexOf(item);
        if (index > -1) {
          processingQueue.splice(index, 1);
        }

        failedItems.delete(item.data.id);

        setTimeout(() => {
          item.wrapper.classList.remove('error');
          item.label.textContent = 'Whitelist';
        }, 3000);
      }
    }
    
    updateQueueIndicator();
    
    if (processingQueue.some(i => !i.processed)) {
      await new Promise(resolve => setTimeout(resolve, RATE_LIMIT_DELAY));
    }
  }
  
  isProcessingQueue = false;
  updateQueueIndicator();
}

function updateQueueIndicator() {
  const indicator = document.getElementById('berry-queue-indicator');
  const queueText = document.getElementById('berry-queue-text');
  
  if (!indicator || !queueText) return;
  
  const pendingCount = processingQueue.filter(i => !i.processed).length;
  const failedCount = failedItems.size;
  
  if (pendingCount > 0 || failedCount > 0) {
    indicator.classList.add('show');
    
    if (pendingCount > 0) {
      queueText.textContent = `Processing ${pendingCount} item${pendingCount !== 1 ? 's' : ''}...`;
    } else if (failedCount > 0) {
      queueText.innerHTML = `${failedCount} item${failedCount !== 1 ? 's' : ''} failed <button class="berry-retry-button" onclick="retryFailedItems()">Retry All</button>`;
    }
  } else {
    indicator.classList.remove('show');
  }
}

window.retryFailedItems = function() {
  failedItems.forEach(item => {
    item.processed = false;
    item.wrapper.classList.remove('error');
    item.wrapper.classList.add('processing');
    item.label.innerHTML = '<span class="berry-loading-spinner"></span>Retrying...';
  });
  
  if (!isProcessingQueue) {
    processQueue();
  }
};

function extractItemData(container) {
  try {
    const link = container.querySelector('a.item-card-link') || 
                 container.querySelector('a[href*="/catalog/"]') || 
                 container.querySelector('a[href*="/bundles/"]');
    
    if (!link || !link.href) {
      return null;
    }
    
    const href = link.href;
    
    let itemId = null;
    let itemType = 'unknown';
    
    const catalogMatch = href.match(/\/catalog\/(\d+)/);
    if (catalogMatch) {
      itemId = catalogMatch[1];
      itemType = 'catalog-item'; 
    }
    
    const bundleMatch = href.match(/\/bundles\/(\d+)/);
    if (bundleMatch) {
      itemId = bundleMatch[1];
      itemType = 'bundle'; 
    }
    
    if (!itemId) {
      return null;
    }
    
    const nameElement = container.querySelector('.item-card-name');
    const priceElement = container.querySelector('.text-robux-tile');
    const creatorElement = container.querySelector('.creator-name');

    // Try multiple selectors to find the thumbnail image
    const imgElement = container.querySelector('.thumbnail-2d-container img') ||
                      container.querySelector('.item-card-thumb-container img') ||
                      container.querySelector('img');

    const itemName = nameElement?.textContent?.trim() ||
                    nameElement?.title ||
                    `Item ${itemId}`;

    // Get the image URL and ensure it's a full URL
    let thumbnailUrl = imgElement?.src || '';

    // If we have a relative URL, make it absolute
    if (thumbnailUrl && !thumbnailUrl.startsWith('http')) {
      thumbnailUrl = new URL(thumbnailUrl, window.location.origin).href;
    }

    return {
      id: itemId,
      robloxItemId: parseInt(itemId),
      itemName: itemName,
      itemType: itemType,
      price: priceElement ? parseInt(priceElement.textContent.replace(/\D/g, '')) : 0,
      thumbnailUrl: thumbnailUrl,
      itemUrl: href,
      creatorName: creatorElement?.textContent?.trim() || ''
    };
  } catch (error) {
    console.error('🍓 Berry Ave: Error extracting item data:', error);
    return null;
  }
}

function showNotification(message, type = 'info') {
  const notification = document.createElement('div');
  notification.className = `berry-notification berry-notification-${type}`;
  notification.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background: white;
    padding: 16px 24px;
    border-radius: 12px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    font-weight: 600;
    z-index: 10000;
    transform: translateX(400px);
    transition: transform 0.3s;
    max-width: 350px;
    border-left: 4px solid ${type === 'success' ? '#10b981' : '#ef4444'};
    color: ${type === 'success' ? '#065f46' : '#991b1b'};
  `;
  notification.textContent = message;
  
  document.body.appendChild(notification);
  setTimeout(() => notification.style.transform = 'translateX(0)', 10);
  setTimeout(() => {
    notification.style.transform = 'translateX(400px)';
    setTimeout(() => notification.remove(), 300);
  }, 5000);
}

async function updateStorageStats(newApprovals = 0) {
  try {
    const result = await chrome.storage.local.get(['stats']);
    const stats = result.stats || { today: 0, total: 0, session: 0, page: 0 };
    
    if (newApprovals > 0) {
      stats.today += newApprovals;
      stats.total += newApprovals;
      stats.session += newApprovals;
    }
    
    await chrome.storage.local.set({ stats });
    
    chrome.runtime.sendMessage({ action: 'statsUpdated', stats }).catch(() => {});
  } catch (error) {
    console.error('🍓 Berry Ave: Error updating stats:', error);
  }
}

function observePageChanges() {
  console.log('🍓 Berry Ave: Setting up mutation observer...');
  const observer = new MutationObserver((mutations) => {
    const hasNewItems = mutations.some(mutation => {
      return Array.from(mutation.addedNodes).some(node => {
        return node.nodeType === 1 && (
          node.classList?.contains('catalog-item-container') ||
          node.querySelector?.('.catalog-item-container')
        );
      });
    });
    
    if (hasNewItems) {
      clearTimeout(window.berryObserverTimeout);
      window.berryObserverTimeout = setTimeout(() => {
        console.log('🍓 Berry Ave: New items detected, processing...');
        addItemCheckboxes();
        checkWhitelistedItems(); 
      }, 500);
    }
  });
  
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getQueueStatus') {
    sendResponse({ 
      queueLength: processingQueue.length,
      failedCount: failedItems.size,
      isProcessing: isProcessingQueue
    });
  } else if (request.action === 'apiKeyUpdated') {
    chrome.storage.local.get(['apiKey']).then(result => {
      apiKey = result.apiKey;
      console.log('🍓 Berry Ave: API key updated');
      // Re-check whitelisted items with new API key
      checkWhitelistedItems();
    });
  } else if (request.action === 'apiKeyRemoved') {
    apiKey = null;
    whitelistedItems.clear();
    processingQueue = [];
    failedItems.clear();
    removingItems.clear();
    isProcessingQueue = false;
    console.log('🍓 Berry Ave: API key removed');
    // Reset visual indicators
    document.querySelectorAll('.berry-whitelisted-badge').forEach(badge => badge.remove());
    document.querySelectorAll('.berry-checkbox-wrapper').forEach(wrapper => {
      wrapper.classList.remove('processing', 'success', 'error');
      wrapper.style.display = 'flex';
      const checkbox = wrapper.querySelector('.berry-checkbox');
      const label = wrapper.querySelector('label');
      if (checkbox) {
        checkbox.disabled = false;
        checkbox.checked = false;
      }
      if (label) {
        label.textContent = 'Whitelist';
      }
    });
    document.querySelectorAll('.item-card-container').forEach(card => {
      card.style.border = '';
    });
    updateStatsDisplay();
  }
});

console.log('%c🍓 Berry Ave Whitelist Tool: Ready!', 'color: #667eea; font-weight: bold; font-size: 14px');