
const WORKER_URL = 'https://berryitems.com';
const DASHBOARD_URL = 'https://berryitems.com';

let elements = {};

document.addEventListener('DOMContentLoaded', initialize);

async function initialize() {
  console.log('Popup initializing...');
  
  elements = {
    apiKeySection: document.getElementById('apiKeySection'),
    statsSection: document.getElementById('statsSection'),
    connectionStatus: document.getElementById('connectionStatus'),
    usernameDisplay: document.getElementById('usernameDisplay'),
    apiKeyInput: document.getElementById('apiKey'),
    saveApiKeyBtn: document.getElementById('saveApiKey'),
    openDashboardBtn: document.getElementById('openDashboard'),
    disconnectBtn: document.getElementById('disconnect'),
    apiMessage: document.getElementById('apiMessage'),
    selectionInfo: document.getElementById('selectionInfo'),
    selectedCount: document.getElementById('selectedCount'),
    todayCount: document.getElementById('todayCount'),
    totalCount: document.getElementById('totalCount'),
    sessionCount: document.getElementById('sessionCount'),
    pageCount: document.getElementById('pageCount')
  };

  showLoadingState();
  
  elements.saveApiKeyBtn.addEventListener('click', () => saveApiKey());
  elements.openDashboardBtn.addEventListener('click', openDashboard);
  if (elements.disconnectBtn) {
    elements.disconnectBtn.addEventListener('click', () => disconnect());
  }
  
  // Check if API key exists after a small delay to ensure storage is ready
  setTimeout(() => {
    checkExistingApiKey();
  }, 100);
}

function showLoadingState() {
  // Keep both sections hidden initially
  elements.apiKeySection.style.display = 'none';
  elements.statsSection.style.display = 'none';
  elements.connectionStatus.textContent = 'Checking...';
  elements.connectionStatus.className = 'status';
}

async function checkExistingApiKey() {
  try {
    console.log('Checking for existing API key...');
    const result = await chrome.storage.local.get(['apiKey', 'userData', 'stats']);
    
    if (result.apiKey) {
      console.log('Found API key, validating...');
      
      if (result.userData) {
        showConnectedState(result.userData);
        if (result.stats) {
          updateStatsDisplay(result.stats);
        }
      }
      
      validateAndUpdateApiKey(result.apiKey);
      
    } else {
      console.log('No API key found');
      showDisconnectedState();
    }
  } catch (error) {
    console.error('Error checking API key:', error);
    showDisconnectedState();
  }
}

async function validateAndUpdateApiKey(apiKey) {
  try {
    const isValid = await validateApiKey(apiKey);
    
    if (!isValid) {
      console.log('Stored API key is invalid');
      await chrome.storage.local.remove(['apiKey', 'userData']);
      showDisconnectedState();
    } else {
      await updateStats();
    }
  } catch (error) {
    console.error('Error validating API key:', error);
  }
}

async function saveApiKey() {
  const apiKey = elements.apiKeyInput.value.trim();
  
  if (!apiKey) {
    showMessage('Please enter an API key', 'error');
    return;
  }

  elements.saveApiKeyBtn.disabled = true;
  elements.saveApiKeyBtn.textContent = 'Connecting...';
  
  try {
    console.log('Validating API key...');
    
    // Make request to validate API key
    const response = await fetch(`${WORKER_URL}/api/auth/validate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-Key': apiKey
      },
      body: JSON.stringify({})
    });

    const contentType = response.headers.get('content-type');
    
    if (!contentType || !contentType.includes('application/json')) {
      throw new Error('Invalid response from server');
    }
    
    const data = await response.json();
    
    if (data.valid) {
      await chrome.storage.local.set({
        apiKey: apiKey,
        userData: data.user
      });
      
      showMessage('Successfully connected!', 'success');
      
      setTimeout(() => {
        showConnectedState(data.user);
        elements.apiKeyInput.value = '';
        updateStats();
      }, 1000);
      
      notifyContentScript('apiKeyUpdated');
      
    } else {
      showMessage(data.error || 'Invalid API key', 'error');
    }
    
  } catch (error) {
    console.error('Error validating API key:', error);
    showMessage(error.message || 'Failed to connect. Please try again.', 'error');
  } finally {
    elements.saveApiKeyBtn.disabled = false;
    elements.saveApiKeyBtn.textContent = 'Connect';
  }
}

async function validateApiKey(apiKey) {
  try {
    const response = await fetch(`${WORKER_URL}/api/auth/validate`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-API-Key': apiKey
      },
      body: JSON.stringify({})
    });
    
    if (response.ok) {
      const data = await response.json();
      if (data.valid && data.user) {
        await chrome.storage.local.set({ userData: data.user });
      }
      return data.valid === true;
    }
    
    return false;
  } catch (error) {
    console.error('Error validating API key:', error);
    return false;
  }
}

function showConnectedState(userData) {
  elements.apiKeySection.style.display = 'none';
  elements.statsSection.style.display = 'block';
  elements.connectionStatus.textContent = 'Connected';
  elements.connectionStatus.classList.remove('disconnected');
  elements.connectionStatus.classList.add('connected');
  
  if (userData && elements.usernameDisplay) {
    elements.usernameDisplay.textContent = `👤 ${userData.username || 'User'} (${userData.role || 'reviewer'})`;
    elements.usernameDisplay.classList.remove('hidden');
  }
}

function showDisconnectedState() {
  elements.apiKeySection.style.display = 'block';
  elements.statsSection.style.display = 'none';
  elements.connectionStatus.textContent = 'Not Connected';
  elements.connectionStatus.classList.remove('connected');
  elements.connectionStatus.classList.add('disconnected');
  
  if (elements.usernameDisplay) {
    elements.usernameDisplay.classList.add('hidden');
  }
}

async function disconnect() {
  try {
    await chrome.storage.local.remove(['apiKey', 'userData', 'stats']);
    showDisconnectedState();
    notifyContentScript('apiKeyRemoved');
  } catch (error) {
    console.error('Error disconnecting:', error);
  }
}

function openDashboard() {
  chrome.tabs.create({ url: DASHBOARD_URL });
}

async function updateStats() {
  try {
    const result = await chrome.storage.local.get(['stats']);
    
    if (result.stats) {
      updateStatsDisplay(result.stats);
    } else {
      updateStatsDisplay({ today: 0, total: 0, session: 0, page: 0 });
    }
    
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab && tab.url && tab.url.includes('roblox.com')) {
        chrome.tabs.sendMessage(tab.id, { action: 'getSelectedCount' }, (response) => {
          if (response && response.count > 0) {
            elements.selectionInfo?.classList.remove('hidden');
            if (elements.selectedCount) {
              elements.selectedCount.textContent = response.count;
            }
          }
        });
      }
    } catch (e) {
    }
    
  } catch (error) {
    console.error('Error updating stats:', error);
  }
}

function updateStatsDisplay(stats) {
  if (elements.todayCount) elements.todayCount.textContent = stats.today || 0;
  if (elements.totalCount) elements.totalCount.textContent = stats.total || 0;
  if (elements.sessionCount) elements.sessionCount.textContent = stats.session || 0;
  if (elements.pageCount) elements.pageCount.textContent = stats.page || 0;
}

function showMessage(text, type) {
  if (!elements.apiMessage) return;
  
  elements.apiMessage.className = type === 'error' ? 'error-message' : 'success-message';
  elements.apiMessage.textContent = text;
  elements.apiMessage.style.display = 'block';
  
  setTimeout(() => {
    if (elements.apiMessage) {
      elements.apiMessage.style.display = 'none';
    }
  }, 5000);
}

async function notifyContentScript(action) {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    
    if (tab && tab.url && tab.url.includes('roblox.com')) {
      chrome.tabs.sendMessage(tab.id, { action: action }).catch(() => {
      });
    }
  } catch (error) {
    console.error('Error notifying content script:', error);
  }
}

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'statsUpdated' && request.stats) {
    updateStatsDisplay(request.stats);
  }
});